// Engine.c
//
// Copyright 2000 Zillions Development
//
// Sample DLL plug-in engine for Zillions of Games (www.zillions-of-games.com)
// Plays a random game of 19x19 Go-Moku


#include <stdio.h>
#include <time.h>
#include "Engine.h"

#define BOARDSIZE		19
#define NUM_POSITIONS	BOARDSIZE * BOARDSIZE

enum {
	WHITE = 0,
	BLACK = 1,
	EMPTY = 2
};

static char *(gstrSideNames[2]) = {"White", "Black"};

static int gColor;
static int gBoard[NUM_POSITIONS];

int StringToPosition(char *s);
void PositionToString(int position, char *s);
void MoveToString(int position, int color, char *s);
long EvaluateMove(int i, long depth);



// StringToPosition
//
// Converts string "s" and returns the position index

int StringToPosition(char *s)
{
	int position = (s[0] - 'A') * BOARDSIZE;
	if (strlen(s) <= 2)
		position += (s[1] - '1');
	else
		position += 10 + (s[2] - '1');
	return position;
}


// PositionToString
//
// The position string is returned in string buffer "s"

void PositionToString(int position, char *s)
{
	int h, v;
	h = position / BOARDSIZE;
	v = position - h*BOARDSIZE + 1;

	*(s++) = 'A' + h;

	if (v >= 10) {
	    *(s++) = '1';
		v -= 10;
	}
	*(s++) = '0' + v;
	*s = '\0';
}


// MoveToString
//
// The move string is returned in string buffer "s"

void MoveToString(int position, int color, char *s)
{
	char p[4];
	PositionToString(position, p);
	sprintf(s, "%s stone %s", gstrSideNames[color], p);
}


// EvaluateMove
//
// Searches move i to the given depth.
// (Actually just returns a random score)

long EvaluateMove(int i, long depth)
{
	// During the search, we must dispatch messages so that
	// the search information is properly updated in the
	// search bar and so the engine will halt when the
	// "Stop Searching" button is pressed.
	MSG msg;
	while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	// Let's pretend our searching took 5 milliseconds
	Sleep(5); // You wouldn't do this in your real engine! 

	// Return a random score
	return rand() - rand();
}

// DLL_Search
//
// The DLL should search from the current position. If it returns DLL_OK it should
// also return the best move found in str; however, it should not make the move
// internally. A separate call to MakeAMove() will follow to make the move the
// engine returns.
//
// -> lSearchTime: Target search time in milliseconds
// -> lDepthLimit: Maximum moves deep the engine should search
// -> lVariety: Variety setting for engine. 0 = no variety, 10 = most variety
// -> pSearchStatus: Pointer to variable where Zillions will report search status
// -> bestMove: Pointer to a string where engine can report the best move found so far
// -> currentMove: Pointer to a string where engine can report the move being searched
// -> plNodes: Pointer to a long where engine can report # of positions searched so far
// -> plScore: Pointer to a long where engine can report current best score in search
// -> plDepth: Pointer to a long where engine can report current search depth
//
// Returns DLL_OK or a negative error code 

DLL_Result FAR PASCAL DLL_Search(long lSearchTime, long lDepthLimit, long lVariety,
	Search_Status *pSearchStatus, LPSTR bestMove, LPSTR currentMove,
	long *plNodes, long *plScore, long *plDepth)
{
	// Note: this engine ignores lSearchTime and lVariety

	int i;
	long lScore;
	*bestMove = *currentMove = '\0';

	*plScore = UNKNOWN_SCORE; // best score so far
	for (*plDepth = 1; *plDepth <= lDepthLimit; ++*plDepth) {
		for (i = 0; i <= NUM_POSITIONS; ++i) {
			if (gBoard[i] == EMPTY) {
				// Search move to position i to the given depth
				MoveToString(i, gColor, currentMove);
				lScore = EvaluateMove(i, *plDepth);
				if (lScore > *plScore) {
					*plScore = lScore;
					strcpy(bestMove, currentMove);
				}
				++(*plNodes);
			}
			if (*pSearchStatus != kKEEPSEARCHING)
				return DLL_OK;
		}
	}
	return DLL_OK;
}


// DLL_MakeAMove
//
// The DLL should try to make the given move internally.
//
// -> move: notation for the move that the engine should make
//
// Returns DLL_OK or a negative error code 

DLL_Result FAR PASCAL DLL_MakeAMove(LPCSTR move) //tries to make the
{
	// Convert string to move and make it
	char a[1000], b[1000], c[1000], d[1000];
	int position, occupant;

	if (*move == '(') { // A board edit
		strcpy(a, &move[1]);
		a[strlen(a)-1] = '\0';
		if (*a == 'x') {
			position = StringToPosition(&a[2]); // x M7
			occupant = EMPTY;
		} else {
			sscanf(a,"%s %s %s", b, c, d); // Black stone M7
			position = StringToPosition(d);
			occupant = !strcmp(b, gstrSideNames[BLACK]) ? BLACK : WHITE;
		}
	} else {
		sscanf(move,"%s %s %s", b, c, d); // e.g. "White Stone H9"
		position = StringToPosition(d);
		occupant = gColor;
		gColor = 1 - gColor; // Switch sides
	}

	gBoard[position] = occupant;
	return DLL_OK;
}


// DLL_StartNewGame
//
// The DLL should reset the gBoard for a new game.
//
// -> variant: The variant to be played as it appears in the variant menu
//
// Returns DLL_OK, DLL_OK_DONT_SEND_SETUP, DLL_OUT_OF_MEMORY_ERROR, or
//   DLL_GENERIC_ERROR

DLL_Result FAR PASCAL DLL_StartNewGame(LPCSTR variant)
{
	// Note: ignores variant name since we only support one variant

	int i;
	gColor = BLACK;

	// Clear the board
	for (i=0; i<NUM_POSITIONS; i++)
		 gBoard[i] = EMPTY;

	return DLL_OK;
}


// DLL_CleanUp
//
// The DLL should free memory and prepare to be unloaded.
//
// Returns DLL_OK, DLL_OUT_OF_MEMORY_ERROR, or DLL_GENERIC_ERROR

DLL_Result FAR PASCAL DLL_CleanUp()
{
	// Prepare to quit, clean up allocated storage
	return DLL_OK;
}


#include <time.h>
int FAR PASCAL LibMain(HANDLE hInstance, WORD wDataSeg, WORD wHeapSize)
{
	// Put engine initialization stuff here
    srand( (unsigned)time( NULL ) ); // Seed the random number generator
    return 1;
}

int FAR PASCAL WEP(void)
{
    return 1;
}